//*************************************************************************************************
//
//	Description:
//		ground_transition.fx - Shader used for transitions between two ground-cover materials. Supports
//													 two materials, each with two diffuse textures with seperate mapping
//													 channels, plus specular and normal maps matching one of the diffuse
//													 textures. Vertex colours are multiplied into the base colour also.
//													 The materials are blended using a third map, which has its own independent
//													 mapping channel.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     05/31/2006  0.1           Created
//		TNettleship     06/08/2006  0.2           Added seperate U/V scaling factors for layer 3.
//		TNettleship			06/27/2006	1.0						Added support for compressed normal maps. Changes
//																							to get it working in the engine.
//		TNettleship			10/03/2006	1.1						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/17/2006	1.2						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.3						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.4						Fixed a bug with directional lighting.
//		TNettleship			11/21/2006	1.5						Added support for deferred rendering.
//		TMann						11/23/2006	1.10					Added _WIN_GL_ support
//		TMann						11/27/2006	1.11					Texture filtering for GL
//		TNettleship			12/04/2006	1.12					Added premultiplied alpha code to the fragment shader.
//		TMann						12/11/2006	1.13					PS3 version
//		TNettleship			01/02/2007	1.14					Stopped premultiplied alpha from being used in opaque
//																							materials.
//		TMann						01/09/2007	1.15					Added PS3/GL literal param passing
//		TNettleship			01/31/2007	1.16					Pixel shader optimisations
//		TNettleship			02/13/2007	1.17					Fixed normal map inconsistency between max & engine
//		BIrvine					02/13/2007	1.18					Temporary shadow implementation
//		TNettleship			04/18/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias, added anisotropy to middle + detail diffuse.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define PROPER_ADDITIVE
#define NORMAL_MAPPING_1
#define NORMAL_MAPPING_2
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif


//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool export = false;
> = 0;

int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 4;
	int RuntimeTexcoord = 3;
	bool export = false;
> = 0;

int texcoord5 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 5;
	int MapChannel = 5;
	int RuntimeTexcoord = 4;
	bool export = false;
> = 0;

#endif


//
// Textures
//

//
// Material 1
//

#ifdef _3DSMAX_
texture broadDiffuseTexture1 : DiffuseMap			// Broad diffuse colour in RGB
#else
texture broadDiffuseTexture1 : TEXTURE				// Broad diffuse colour in RGB
#endif
<
	string UIName = "Broad Diffuse 1";
	bool appEdit = true;
>;

texture middleDiffuseTexture1 : TEXTURE				// Middle diffuse colour in RGB
<
	string UIName = "Middle Diffuse 1";
	bool appEdit = true;
>;

texture detailDiffuseTexture1 : TEXTURE				// Detail diffuse colour in RGB
<
	string UIName = "Detail Diffuse 1";
	bool appEdit = true;
>;

float detailUScale1														// Detail layer U scale
<
	string UIName = "Detail 1 Layer U Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

float detailVScale1														// Detail layer V scale
<
	string UIName = "Detail 1 Layer V Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;


bool additiveDetail1													// Additive detail?
<
	string UIName = "Additive Detail 1?";
	bool appEdit = true;
> = false;

SPECIALISATION_PARAM( properAdditive, "Proper Additive?", "PROPER_ADDITIVE" )	// TRUE if true additive detail should be used

texture specularTexture1 : TEXTURE						// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture 1";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap1, "Use normal map 1?", "NORMAL_MAPPING_1" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency1, tangent, TANGENT, useNormalMap1 )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency1, binormal, BINORMAL, useNormalMap1 )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 )
DEPENDENT_TEXTURE_PARAM( normalTexture1, "Normal Texture 1", useNormalMap1 )
#endif


//
// Material 2
//

texture broadDiffuseTexture2 : TEXTURE				// Broad diffuse colour in RGB
<
	string UIName = "Broad Diffuse 2";
	bool appEdit = true;
>;

texture middleDiffuseTexture2 : TEXTURE				// Middle diffuse colour in RGB
<
	string UIName = "Middle Diffuse 2";
	bool appEdit = true;
>;

texture detailDiffuseTexture2 : TEXTURE				// Detail diffuse colour in RGB
<
	string UIName = "Detail Diffuse 2";
	bool appEdit = true;
>;

float detailUScale2														// Detail layer U scale
<
	string UIName = "Detail 2 Layer U Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

float detailVScale2														// Detail layer V scale
<
	string UIName = "Detail 2 Layer V Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

bool additiveDetail2													// Additive detail?
<
	string UIName = "Additive Detail 2?";
	bool appEdit = true;
> = false;

texture specularTexture2 : TEXTURE						// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture 2";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap2, "Use normal map 2?", "NORMAL_MAPPING_2" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency2, tangent, TANGENT, useNormalMap2 )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency2, binormal, BINORMAL, useNormalMap2 )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_2 )
DEPENDENT_TEXTURE_PARAM( normalTexture2, "Normal Texture 2", useNormalMap2 )
#endif


//
// Blend material
//

texture blendTexture : TEXTURE								// Blend amount in G
<
	string UIName = "Blend Map";
	bool appEdit = true;
>;

bool blendInvert															// Blend invert?
<
	string UIName = "Blend Invert?";
	bool appEdit = true;
> = false;


float minSpecPower1
<
	string UIName = "Min Specular Power L1";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower1
<
	string UIName = "Max Specular Power L1";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor1
<
	string UIName = "Specular Factor L1";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float minSpecPower2
<
	string UIName = "Min Specular Power L2";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower2
<
	string UIName = "Max Specular Power L2";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor2
<
	string UIName = "Specular Factor L2";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D broadDiffuseMap1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="broadDiffuseTexture1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < broadDiffuseTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D middleDiffuseMap1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="middleDiffuseTexture1"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < middleDiffuseTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D detailDiffuseMap1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="detailDiffuseTexture1"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < detailDiffuseTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D specularMap1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 )
sampler2D normalMap1 : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

sampler2D broadDiffuseMap2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="broadDiffuseTexture2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < broadDiffuseTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D middleDiffuseMap2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="middleDiffuseTexture2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < middleDiffuseTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D detailDiffuseMap2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="detailDiffuseTexture2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < detailDiffuseTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D specularMap2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture2"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_2 )
sampler2D normalMap2 : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture2"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

sampler2D blendMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blendTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < blendTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 1 texture coord
	float2 texCoord2 : TEXCOORD3;													// UV channel 2 texture coord
	float2 texCoord3 : TEXCOORD4;													// UV channel 3 texture coord
	float2 texCoord4 : TEXCOORD5;													// UV channel 4 texture coord
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 1 texture coord
	float2 texCoord2 : TEXCOORD2;													// UV channel 2 texture coord
	float2 texCoord3 : TEXCOORD3;													// UV channel 3 texture coord
	float2 texCoord4 : TEXCOORD4;													// UV channel 4 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space normal
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 and 1 (packed) -- material 1 broad and middle
	float4 texCoord23	: TEXCOORD1;												// UV coords for texture channel 2 and 3 (packed) -- material 2 broad and middle
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float3 normal			: TEXCOORD3;												// Normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
		// If normal mapping is enabled, we need to squeeze a tangent and binormal vector
		// into the interpolators. We pack the binormal into the texcoord4 interpolator and the tangent interpolator
		float4 texCoord4_plus_binormal_xy : TEXCOORD5;			// UV coords for texture channel 4, packed with the xy of the binormal vector in zw
		float4 tangent_plus_binormal_z		: TEXCOORD6;			// Tangent vector (world space), plus binormal.z in the w component

		DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
		// With no normal mapping, we don't need to scrunch anything
		float2 texCoord4  : TEXCOORD5;											// UV coords for texture channel 4
		DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT GroundTransitionVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord01 = float4( _input.texCoord0.xy, _input.texCoord1.xy );
	_output.texCoord23 = float4( _input.texCoord2.xy, _input.texCoord3.xy );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;

	DEPENDENT_CODE_START_OR( useNormalMap1, useNormalMap2 )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
		float3 worldCoordsBinormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
		float3 worldCoordsTangent = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.texCoord4_plus_binormal_xy = float4( _input.texCoord4.xy, worldCoordsBinormal.xy );
		_output.tangent_plus_binormal_z		 = float4( worldCoordsTangent, worldCoordsBinormal.z );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap1 )
#if defined( _3DSMAX_ )
		_output.texCoord4_plus_binormal_xy = float4( _input.texCoord4.xy, 0.0f, 0.0f );
#elif !( defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 ) )
		_output.texCoord4 = _input.texCoord4.xy;
#endif
	DEPENDENT_CODE_END_OR( useNormalMap1, useNormalMap2 )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec.xyz, 0);

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour											: TEXCOORD4;						// Vertex colour
	float4 texCoord01									: TEXCOORD0;				// UV coords for texture channels 0 and 1 (packed) -- material 1 broad and middle
	float4 texCoord23									: TEXCOORD1;				// UV coords for texture channel 2 and 3 (packed) -- material 2 broad and middle
	float4 eye												: TEXCOORD2;				// Eye vector (world space)
	float3 normal											: TEXCOORD3;				// Normal
	float4 texCoord4_plus_binormal_xy : TEXCOORD5;				// UV coords for texture channel 4, packed with the xy of the binormal vector in zw
	float4 tangent_plus_binormal_z		: TEXCOORD6;				// Tangent vector (world space), plus binormal.z in the w component

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4_centroid;							// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 and 1 (packed) -- material 1 broad and middle
	float4 texCoord23	: TEXCOORD1;												// UV coords for texture channel 2 and 3 (packed) -- material 2 broad and middle
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
	float3 normal     : TEXCOORD3_centroid;								// Normal vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
		// If normal mapping is enabled, we need to squeeze a tangent and binormal vector
		// into the interpolators. We pack the binormal into the texcoord4 interpolator and the tangent interpolator
		float4 texCoord4_plus_binormal_xy : TEXCOORD5_centroid;	// UV coords for texture channel 4, packed with the xy of the binormal vector in zw
		float4 tangent_plus_binormal_z		: TEXCOORD6_centroid;	// Tangent vector (world space), plus binormal.z in the w component

		DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
		// With no normal mapping, we don't need to scrunch anything
		float2 texCoord4  : TEXCOORD5;												// UV coords for texture channel 4
		DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};


//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT GroundTransitionFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read blend layer texture
	float blendFactor;
	DEPENDENT_CODE_START_OR( useNormalMap1, useNormalMap2 )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
		blendFactor = tex2D( blendMap, _input.texCoord4_plus_binormal_xy.xy ).g;
#endif
	DEPENDENT_CODE_ELSE( useNormalMap1 )
#if defined( _3DSMAX_ )
		blendFactor = tex2D( blendMap, _input.texCoord4_plus_binormal_xy.xy ).g;
#elif !( defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 ) )
		blendFactor = tex2D( blendMap, _input.texCoord4 ).g;
#endif
	DEPENDENT_CODE_END_OR( useNormalMap1, useNormalMap2 )

	if ( blendInvert )
	{
		blendFactor = 1.0f - blendFactor;
	}

	// Read layer 1 textures
	float2 detail1Coord = _input.texCoord01.zw * float2( detailUScale1, detailVScale1 );
	float4 broadDiffuseTexColour1 = tex2D( broadDiffuseMap1, _input.texCoord01.xy );
	float4 middleDiffuseTexColour1 = tex2D( middleDiffuseMap1, _input.texCoord01.zw );
	float4 detailDiffuseTexColour1 = tex2D( detailDiffuseMap1, detail1Coord );
	float4 specularTexColour1 = tex2D( specularMap1, detail1Coord );

	// Calculate layer 1 base colour
	float4 baseColour1;
	if ( additiveDetail1 )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			baseColour1 = ( broadDiffuseTexColour1 * middleDiffuseTexColour1 ) + detailDiffuseTexColour1;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			baseColour1 = ( ( broadDiffuseTexColour1 * middleDiffuseTexColour1 ) + detailDiffuseTexColour1 ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		baseColour1 = broadDiffuseTexColour1 * middleDiffuseTexColour1 * detailDiffuseTexColour1;
	}

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );

	float3 tangent;
	float3 binormal;
	// If either layer requires normal mapping
	DEPENDENT_CODE_START_OR( useNormalMap1, useNormalMap2 )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 ) || defined( NORMAL_MAPPING_2 )
		// Normalise the input tangent & binormal vectors
		tangent = normalize( float3( _input.tangent_plus_binormal_z.xyz ) );
		binormal = normalize( float3( _input.texCoord4_plus_binormal_xy.zw, _input.tangent_plus_binormal_z.w ) );
#endif
	DEPENDENT_CODE_END_OR( useNormalMap1, useNormalMap2 )

	// Do layer 1 normal mapping if required
	float3 normal1;
	DEPENDENT_CODE_START( useNormalMap1 )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_1 )
		// Fetch and decode the map normal
		float4 normalMapColour1 = tex2D( normalMap1, detail1Coord );
		float3 normalFromMap1 = ( normalMapColour1.rgb * 2.0f ) - 1.0f;

		// Perturb the tangent space normal by the normal map
		normal1 = ( TSnormal * normalFromMap1.z ) + ( normalFromMap1.x * binormal ) + ( normalFromMap1.y * tangent );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap1 )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING_1 )
		// No normal map, so use interpolated normal
		normal1 = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap1 )

	// Read layer 2 textures
	float2 detail2Coord = _input.texCoord23.zw * float2( detailUScale2, detailVScale2 );
	float4 broadDiffuseTexColour2 = tex2D( broadDiffuseMap2, _input.texCoord23.xy );
	float4 middleDiffuseTexColour2 = tex2D( middleDiffuseMap2, _input.texCoord23.zw );
	float4 detailDiffuseTexColour2 = tex2D( detailDiffuseMap2, detail2Coord );
	float4 specularTexColour2 = tex2D( specularMap2, detail2Coord );

	// Calculate layer 2 base colour
	float4 baseColour2;
	if ( additiveDetail2 )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			baseColour2 = ( broadDiffuseTexColour2 * middleDiffuseTexColour2 ) + detailDiffuseTexColour2;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			baseColour2 = ( ( broadDiffuseTexColour2 * middleDiffuseTexColour2 ) + detailDiffuseTexColour2 ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		baseColour2 = broadDiffuseTexColour2 * middleDiffuseTexColour2 * detailDiffuseTexColour2;
	}

	// If layer 2 normal map support is required
	float3 normal2;
	DEPENDENT_CODE_START( useNormalMap2 )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING_2 )
		// Fetch and decode the map normal
		float4 normalMapColour2 = tex2D( normalMap2, detail2Coord );
		float3 normalFromMap2 = ( normalMapColour2.rgb * 2.0f ) - 1.0f;

		// Perturb the tangent space normal by the normal map
		normal2 = ( TSnormal * normalFromMap2.z ) + ( normalFromMap2.x * binormal ) + ( normalFromMap2.y * tangent );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap2 )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING_2 )
		// No normal map, so use interpolated normal
		normal2 = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap2 )

	// Blend the two submaterials' colours, normals and parameters into one then use it to perform lighting
	float4 diffuseTexColour = lerp( baseColour1, baseColour2, blendFactor );
	float4 specularTexColour = lerp( specularTexColour1, specularTexColour2, blendFactor );
	float3 normal = normalize( lerp( normal1, normal2, blendFactor ) );
	float minSpecPowerValue = lerp( minSpecPower1, minSpecPower2, blendFactor );
	float maxSpecPowerValue = lerp( maxSpecPower1, maxSpecPower2, blendFactor );
	float globalSpecularFactorValue = lerp( globalSpecularFactor1, globalSpecularFactor2, blendFactor );

	// Apply ambient lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = lerp( broadDiffuseTexColour1.w, broadDiffuseTexColour2.w, blendFactor );
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}





struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float4 texCoord01	: TEXCOORD0;												// UV coords for texture channels 0 and 1 (packed) -- material 1 broad and middle
	float4 texCoord23	: TEXCOORD1;												// UV coords for texture channel 2 and 3 (packed) -- material 2 broad and middle
	float2 texCoord4 : TEXCOORD2;													// UV coords for texture channel 4 (packed) -- blend layer
};



VSOUTPUT_LD GroundTransitionLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord01 = float4( _input.texCoord0.xy, _input.texCoord1.xy );
	_output.texCoord23 = float4( _input.texCoord2.xy, _input.texCoord3.xy );
	_output.texCoord4.xy = _input.texCoord4.xy;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}



PSOUTPUT GroundTransitionLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read blend layer texture
	float blendFactor = tex2D( blendMap, _input.texCoord4.xy ).g;
	if ( blendInvert )
	{
		blendFactor = 1.0f - blendFactor;
	}

	// Read layer 1 textures
	float2 detail1Coord = _input.texCoord01.zw * float2( detailUScale1, detailVScale1 );
	float4 broadDiffuseTexColour1 = tex2D( broadDiffuseMap1, _input.texCoord01.xy );
	float4 middleDiffuseTexColour1 = tex2D( middleDiffuseMap1, _input.texCoord01.zw );
	float4 detailDiffuseTexColour1 = tex2D( detailDiffuseMap1, detail1Coord );

	// Calculate layer 1 base colour
	float4 baseColour1;
	if ( additiveDetail1 )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			baseColour1 = ( broadDiffuseTexColour1 * middleDiffuseTexColour1 ) + detailDiffuseTexColour1;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			baseColour1 = ( ( broadDiffuseTexColour1 * middleDiffuseTexColour1 ) + detailDiffuseTexColour1 ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		baseColour1 = broadDiffuseTexColour1 * middleDiffuseTexColour1 * detailDiffuseTexColour1;
	}

	// Read layer 2 textures
	float2 detail2Coord = _input.texCoord23.zw * float2( detailUScale2, detailVScale2 );
	float4 broadDiffuseTexColour2 = tex2D( broadDiffuseMap2, _input.texCoord23.xy );
	float4 middleDiffuseTexColour2 = tex2D( middleDiffuseMap2, _input.texCoord23.zw );
	float4 detailDiffuseTexColour2 = tex2D( detailDiffuseMap2, detail2Coord );

	// Calculate layer 2 base colour
	float4 baseColour2;
	if ( additiveDetail2 )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			baseColour2 = ( broadDiffuseTexColour2 * middleDiffuseTexColour2 ) + detailDiffuseTexColour2;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			baseColour2 = ( ( broadDiffuseTexColour2 * middleDiffuseTexColour2 ) + detailDiffuseTexColour2 ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		baseColour2 = broadDiffuseTexColour2 * middleDiffuseTexColour2 * detailDiffuseTexColour2;
	}

	// Blend the two submaterials' colours, normals and parameters into one then use it to perform lighting
	float4 diffuseTexColour = lerp( baseColour1, baseColour2, blendFactor );

	// Apply ambient lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;


	accumulatedColour.w = lerp( broadDiffuseTexColour1.w, broadDiffuseTexColour2.w, blendFactor );
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Ground_Transition
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Transition";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Ground_Transition_LowDetail";
	int    lowDetailDeferredID		= 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundTransitionVertexShader();
		PixelShader = compile sce_fp_rsx GroundTransitionFragmentShader();
#else
		VertexShader = compile vs_3_0 GroundTransitionVertexShader();
		PixelShader = compile ps_3_0 GroundTransitionFragmentShader();
#endif
	}
}



technique Ground_Transition_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Transition_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundTransitionLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx GroundTransitionLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 GroundTransitionLowDetailVertexShader();
		PixelShader = compile ps_3_0 GroundTransitionLowDetailFragmentShader();
#endif
	}
}
